/**
 * 定义所有验证器.
 */
define(["jquery",
        "ecp.utils"], function($, ecpUtils) {

	'use strict'; // 使用js语法严格模式(提高代码运行效率)
	/**
	 * 省份编号映射.
	 */
    var PROVINCE_CODE = ['11', '12', '13', '14', '15', '21',
        '22', '23', '31', '32', '33', '34',
        '35', '36', '37', '41', '42', '43',
        '44', '45', '46', '50', '51', '52',
        '53', '54', '61', '62', '63', '64',
        '65', '71', '81', '82'];

	/**
	 * 身份证加权因子表.
	 */
    var CARD_FACTOR = [7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2];

	/**
	 * 校验码换算表.
	 */
    var CARD_PARITY = ["1", "0", "X", "9", "8", "7", "6", "5", "4", "3", "2"];
    
	var ValidatorUtil = function() {};
	ValidatorUtil.prototype = {
		/**
		 * 是否null或空串.
		 */
		isEmpty : function(str) {
			return ecpUtils.isEmpty(str);
		},
		/**
		 * 是否null或者空串或空格
		 */
		isBlank : function(str) {
			if (this.isEmpty(str)) {
				return true;
			}
			var trimVal = $.trim(str);
			return this.isEmpty(trimVal);
		},
		/**
		 * 是否手机号.
		 */
		isMovetel : function(str) {
			var reg = /^1[3456789]\d{9}$/;
			return reg.test(str);
		},
        /**
         * 校验中文名称.
         *
         * @param chineseName 中文姓名.
         * @returns {boolean}
         */
        validateChineseName: function (chineseName) {
            return /^[\u4e00-\u9fa5]{2,10}$/.test(chineseName);
        },
        /**
         * 校验身份证号码.
         *
         * @param cardNo 身份证号码
         * @returns {boolean}
         */
        validateCardNo: function (cardNo) {
            var length = cardNo.length;
            if (18 === length) {
                if (!/^[1-9]\d{5}(18|19|20)\d{2}((0[1-9])|(1[0-2]))(([0-2][1-9])|10|20|30|31)\d{3}[0-9Xx]$/.test(cardNo)) {
                    return false;
                } else {
                    var arr = cardNo.split("");
                    var code = cardNo.substring(17).toUpperCase();
                    var sum = 0;
                    for (var i = 0; i < 17; i++) {
                        var current = parseInt(arr[i]);
                        sum += current * CARD_FACTOR[i];
                    }
                    var realCode = CARD_PARITY[sum % 11];
                    if (code !== realCode) {
                        return false;
                    }
                }
                // 生日格式
                var date = cardNo.substring(6, 14);
                if (!/^(18|19|20)\d{2}((0[1-9])|(1[0-2]))(([0-2][1-9])|10|20|30|31)$/.test(date)) {
                    return false;
                }
                // 省份编号
                var provinceCode = cardNo.substring(0, 2);
                return (PROVINCE_CODE.indexOf(provinceCode) >= 0);
            } else if (15 === length) {
                return /^[1-9]\d{5}\d{2}((0[1-9])|(10|11|12))(([0-2][1-9])|10|20|30|31)\d{2}$/.test(cardNo);
            }
            return false;
        },
		/**
		 * 是否身份证号.
		 */
		isCardId : function(str) {
			if (this.isBlank(str)) {
				return false;
			}
			return this.validateCardNo(str);
		},
		/**
		 * 是否邮箱.
		 */
		isEmail : function(str) {
			var reg = /^[\.A-Za-z0-9\u4e00-\u9fa5]+@[a-zA-Z0-9_-]+(\.[a-zA-Z0-9_-]+)+$/;
			return reg.test(str);
		},
		/**
		 * 是否纯数字.
		 */
		isPureNumber : function(str) {
			return /^[0-9]+$/.test(str);
		},
		/**
		 * 是否汉字.
		 */
		isChinese : function(str) {
			return this.validateChineseName(str);
		},
		
		/**
		 * 校验密码(8-20位密码,由数字、特殊字符、字母组成).
		 * @param password
		 * @returns true 满足要求
		 */
		isPwd : function (password){
			let pwdRegex =  /^(?=.*[a-zA-Z])(?=.*\d)(?=.*[~!@#$%^&*()_+`\-={}:";'<>?,.\/]).{8,20}$/; 
			return pwdRegex.test(password);
		},
		
		/**
		 * 是否有非法字符.
		 */
		isIllegalCharacter : function (str){
			var check = /[/#&'"\\\r\n\t\f\b\s^]/;
			return check.test(str);
		},
		/**
		 *是否是 符合密码规则，太简单的密码.
		 */
		isSimplePwd : function (str){
			return '1234.abcd'== str;
		},
		/**
		 * 是否是统一社会信用代码.
		 */
		isUnifiedSocialCreditCode : function (str){
			var check = /^[^_IOZSVa-z\W]{2}\d{6}[^_IOZSVa-z\W]{10}$/g;
			return check.test(str);
		}
	};

	return new ValidatorUtil();
});